unit Oscillograph02;
// ====================================================================
(*
       
      .   
     TOscillograph.
        ,  
        TOscillograph.
  :
       1.  TItem1 -      .
       2.  TQueue1 - ,     FIFO.

    TOscillograph       
       (TPanel).
        
      ,     ,
           .

         1.02. ()  , , , , 2017.
                     () Source code  ..
          03.01.2018
*)
// ====================================================================

interface

uses Windows, Messages, Classes, Controls, SysUtils, Graphics, Dialogs,
     StdCtrls, ExtCtrls, Menus;

// ====================================================================
//                  TItem1 ( 1.0)
//       
// ====================================================================
type TItem1 = class(TObject)
protected
  fNext   : TItem1;               //    
  fData   : extended;             // ,    
end;
// ====================================================================
//                   TQueue1 ( 1.0)
//   ,     FIFO
// ====================================================================
type TQueue1 = class(TItem1)
private
  fHead     : TItem1;            //    
  fTail     : TItem1;            //     
  fCurrent  : TItem1;            //    
  fCount    : integer;           //    
  fMaxCount : integer;           //    
  procedure   InsertToTail(RqData : extended);
  procedure   CutFromHead();
  procedure   SetQueueMaxCount(RqCount : integer);
public
  constructor Create();
  procedure   AddData (RqData : extended);
  function    SetCurrentItem(Cmd : char) : boolean;
  function    GetCurrentData () : extended;
  procedure   ClearQueue();
  procedure   Free();
  property    QueueMaxCount : integer read fMaxCount write SetQueueMaxCount;
end;
// ====================================================================
//                TOscillograph
//            
// ====================================================================
type TOscillograph = class(TObject)
private
   // -----------------------------
   fTitle       : string;       //  
   // -----------------------------
   fPanelRect   : Trect;        //   
   fPanel       : TPanel;       //  
   fImg         : TImage;       // Image  
   fBMP         : TBitMap;      //  
   fPM          : TPopupMenu;   // PopUpMenu  
   fBeam1       : TQueue1;      // FIFO   
   fLbT         : TLabel;       // Label  
   fLbV         : TLabel;       // Label  
   // -----------------------------
   //  
   fYMes        : string;       //    
   fYMax        : extended;     // .    
   fXMax        : integer;      // .    
   // -----------------------------
   //  
   fDigit       : boolean;      //   
   fdYShow      : boolean;      //    
   fYPrev       : extended;     //   
   fXCount      : integer;      //    
   // -----------------------------
   //  
   fGridY       : extended;     //     Y
   fGridX       : integer;      //     X
   // -----------------------------
   //  
   fBeamColor   : TColor;       //  
   fFonColor    : TColor;       //   
   fGridColor   : TColor;       //    
   fTextColor   : TColor;       //    
   // -----------------------------
   //   
   fAlarmOn     : boolean;      //    
   fY1Alarm     : extended;     //    1
   fY2Alarm     : extended;     //    2
   fc1Alarm     : TColor;       //     1
   fc2Alarm     : TColor;       //     2
   // -----------------------------
   fRecording   : boolean;      //     
   fFileNumPnt  : integer;      //       
   fFileCount   : integer;      //     
   fListFilePnt : TStringList;  //      
   // -----------------------------
   //  property
   procedure SetTitle(RqTitle : string);
   procedure SetYMax (RqYMax  : extended);
   procedure SetXMax (RqXMax  : integer);
   // -----------------------------
   //    PopupMenu
   procedure MenuClick(Sender : TObject);
   // -----------------------------
   //     fImg
   procedure BmpToImg ();
   //      
   procedure ShowGrids();
   //  
   procedure ImageClear(RqColor : TColor);
   //      X
   function  CalcX() : integer;
   //      Y
   function  CalcY(RqData : extended) : integer;
   //    
   function GetBeamColor(RqData : extended): TColor;
   //    
   procedure ShowFirstPoint(RqData : extended);
   //    
   procedure ShowNextPoint(RqData : extended);
    // -----------------------------
   //       
   procedure SetFileNumPoint();
   //       
   procedure DialogSaveToFile();
   //      
   procedure AddDataToList(RqData : extended);
   //  
   procedure ShowTitle();
public
   // -----------------------------
   //        
   constructor Create(RqParent : TWinControl; RqPanelRect : TRect);
   procedure   Free;
   // -----------------------------
   //    
   procedure AddNewData (RqData : extended);
   //   
   procedure ShowBeam();
   //  
   procedure Clear();
   // -----------------------------
   //  
   property Title      : string   read fTitle     write SetTitle;
   //   
   property PanelRect  : Trect    read fPanelRect;
    //    
   property  YMes      : string   read fYMes     write fYMes;
   //   
   property YMax       : extended read fYMax      write SetYMax;
   //      X
   property XMax       : integer  read fXMax      write SetXMax;
   //   
   property Digit      : boolean  read fDigit     write fDigit;
   //   
   property BeamColor  : TColor   read fBeamColor write fBeamColor;
   //   
   property FonColor   : TColor   read fFonColor  write fFonColor;
   //   
   property GridColor  : TColor   read fGridColor write fGridColor;
   //    
   property TextColor  : TColor   read fTextColor write fTextColor;
   //     Y (   )
   property GridY      : extended read fGridY     write fGridY;
   //     X ( )
   property GridX      : integer  read fGridX     write fGridX;
   //    (   )
   property Y1Alarm    : extended read fY1Alarm   write fY1Alarm;
   //    (   )
   property Y2Alarm    : extended read fY2Alarm   write fY2Alarm;
   //     
   property Alarm1Color  : TColor read fc1Alarm   write fc1Alarm;
   //     
   property Alarm2Color  : TColor read fc2Alarm   write fc2Alarm;

end;

// ====================================================================
// ====================================================================
implementation
// ====================================================================
// ====================================================================


// ====================================================================
//                   TQueue1  ( 1.0)
//   ,     FIFO
// ====================================================================
// --------------------------------------------------------------------
//   ,        
procedure TQueue1.InsertToTail(RqData : extended);
var  wp : TItem1;
begin
   wp := TItem1.Create;
   wp.fNext := nil;
   wp.fData := RqData;
   if not Assigned(fTail)
   then begin  //     
      fTail    := wp;
      fHead    := fTail;
      fCurrent := fTail;
      fCount   := 1;
   end
   else begin  //     
      fTail.fNext := wp;
      fTail := wp;
      fCount := fCount + 1;
   end;
end;
// --------------------------------------------------------------------
//     
procedure TQueue1.CutFromHead();
var  wp  :  TItem1;
begin
   if not Assigned(fHead)
   then begin
      //  
      fTail    := nil;
      fCurrent := nil;
      fCount   := 0;
   end
   else begin
      //     
      wp := fHead;
      if not Assigned(wp.fNext)
      then begin   //       
          fHead    := nil;
          fTail    := nil;
          fCurrent := nil;
          fCount   := 0;
      end
      else begin   //    
        fHead := wp.fNext;
        //     
        if fCurrent = wp then fCurrent := fHead;
        fCount := fCount - 1;
        if fCount < 1 then ShowMessage(' Queue');
      end;
      //   
      wp.Free;
   end;
end;
// --------------------------------------------------------------------
//  
constructor TQueue1.Create();
begin
   inherited Create;
   fHead     := nil;   //  
   fTail     := nil;   //  
   fCurrent  := nil;   //  
   fCount    := 0;     //  
   fMaxCount := 10;    //   
end;
// --------------------------------------------------------------------
//     
procedure TQueue1.AddData (RqData : extended);
begin
   InsertToTail(RqData);
   if fCount > fMaxCount then CutFromHead();
end;
// --------------------------------------------------------------------
//    
function TQueue1.SetCurrentItem(Cmd : char) : boolean;
begin
    Result   := False;
    if not Assigned(fHead)
    then begin
        fCurrent := nil;
    end
    else begin
       //   
       case Cmd of
       'H' : begin    //     () 
                fCurrent := fHead;
                Result := True;
             end;
       'N' : begin   //      
                if Assigned(fCurrent)
                then begin
                   fCurrent := fCurrent.fNext;
                   if Assigned(fCurrent) then Result := True;
                end
                else fCurrent := nil;
            end;
       end;
    end;
end;
// --------------------------------------------------------------------
//      
function  TQueue1.GetCurrentData () : extended;
begin
   Result := 0;
   if Assigned(fCurrent) then Result := fCurrent.fData;
end;
// --------------------------------------------------------------------
//     
procedure TQueue1.ClearQueue();
begin
   while Assigned(fHead) do CutFromHead();
end;
// --------------------------------------------------------------------
//        
procedure TQueue1.SetQueueMaxCount(RqCount : integer);
begin
   if RqCount > 0
   then begin
      ClearQueue();
      fMaxCount := RqCount;
   end;
end;
// --------------------------------------------------------------------
//  
procedure  TQueue1.Free();
begin
  ClearQueue();
  inherited Free();
end;

// ====================================================================
//                TOscillograph
//            
// ====================================================================
const SubScrLen = 50;               //      Y
      MinPanelW = 200 + SubScrLen;  // .   
      MinPanelH = 120;              // .   
      RightOff  = 5;                //    

// --------------------------------------------------------------------
//   
constructor TOscillograph.Create(RqParent : TWinControl; RqPanelRect : TRect);
var MenuItems : array of TMenuItem;
begin
   inherited Create;
   fPanelRect := RqPanelRect;
   with RqPanelRect do
   begin
      if Left < 0 then fPanelRect.Left := 0;
      if Top < 0 then  fPanelRect.Top  := 0;
      if Right - fPanelRect.Left < MinPanelW
      then fPanelRect.Right := fPanelRect.Left + MinPanelW;
      if Bottom - fPanelRect.Top < MinPanelH
      then fPanelRect.Bottom := fPanelRect.Top + MinPanelH;
   end;
   // -----------------------------
   //   
   fPanel := TPanel.Create(RqParent);
   fPanel.Parent := RqParent;
   fPanel.Visible := False;
   with fPanelRect do fPanel.SetBounds(Left,Top,Right-Left,Bottom-Top);
   // -----------------------------
   //  Image  
   fImg  := TImage.Create(fPanel);
   fImg.Parent := fPanel;
   fImg.SetBounds(2,20,fPanel.Width-4,fPanel.Height-40);
   // -----------------------------
   //   
   fBMP := TBitMap.Create;
   fBMP.PixelFormat := pf24bit;
   fBMP.Height := fImg.Height;
   fBMP.Width  := fImg.Width;
    // -----------------------------
   //  Label    
   fLbV := TLabel.Create(fPanel);
   fLbV.Parent := fPanel;
   fLbV.SetBounds(2,fPanel.Height-18,fPanel.Width-4,16);
   fLbV.AutoSize := False;
   // -----------------------------
   //  Label   
   fLbT := TLabel.Create(fPanel);
   fLbT.Parent := fPanel;
   fLbT.SetBounds(2,3,fPanel.Width-4,16);
   fLbT.AutoSize := False;
   fLbT.Alignment := taLeftJustify;
   // -----------------------------
   fPanel.Visible := True;
   // -----------------------------
   //    (PopupMenu)
   SetLength(MenuItems, 6);
   MenuItems[0]:= NewItem(' ', TextToShortCut(''),
                          True, True, MenuClick, 0, 'M_ZONE');
   MenuItems[0].Tag := 1;
   // --------------------
   MenuItems[1]:= NewItem(' ', TextToShortCut(''),
                          False, True, MenuClick, 0, 'M_DIGIT');
   MenuItems[1].Tag := 2;
   // --------------------
   MenuItems[2]:= NewItem('', TextToShortCut(''),
                          False, True, MenuClick, 0, 'M_INC');
   MenuItems[2].Tag := 3;
   // --------------------
   MenuItems[3]:= NewItem('-', TextToShortCut(''),
                          False, True, MenuClick, 0, 'M_SEP');
   // --------------------
   MenuItems[4]:= NewItem('     ', TextToShortCut(''),
                          False, True, MenuClick, 0, 'M_SET_PNT');
   MenuItems[4].Tag := 4;
   MenuItems[5]:= NewItem('    ', TextToShortCut(''),
                          False, True, MenuClick, 0, 'M_RUN_PNT');
   MenuItems[5].Tag := 5;
   // --------------------
   //    (PopupMenu)
   fPM := NewPopupMenu(fImg, 'Menu',
                       paLeft, True, MenuItems);
   SetLength(MenuItems,0);
   fImg.PopupMenu := fPM;
   // -----------------------------
   //  FIFO    
   fBeam1  := TQueue1.Create;
   fXMax   := 50;
   fBeam1.QueueMaxCount := fXMax;
   // -----------------------------
   //   
   fYMax  := 100;
   // -----------------------------
   //    
   fGridY := fYMax / 5;
   fGridX := fXMax div 5;
   // -----------------------------
   fDigit       := False;         //    
   // -----------------------------
   //    
   fBeamColor   := clLime;
   fFonColor    := RGB(64,64,64);
   fGridColor   := clGray;
   fTextColor   := clWhite;
   ImageClear(fFonColor);
   // -----------------------------
   //  
   fAlarmOn     := True;           //    
   fY1Alarm     := fYMax * 0.7;    //    1
   fY2Alarm     := fYMax * 0.9;    //    2
   fc1Alarm     := clYellow;       //      
   fc2Alarm     := clRed;          //      
   // -----------------------------
   fRecording   := False;          //     
   fFileNumPnt  := 200;            //       
   fFileCount   := 0;              //     
   //      
   fListFilePnt := TStringList.Create;
   //  
   ShowTitle();
end;

// --------------------------------------------------------------------
procedure TOscillograph.Free;
begin
   if Assigned(fListFilePnt)
   then begin
     fListFilePnt.Free;
     fListFilePnt := nil;
   end;
   if Assigned(fBeam1)   then fBeam1.Free;
   if Assigned(fPM)      then fPM.Free;
   if Assigned(fLbV)     then fLbV.Free;
   if Assigned(fLbT)     then fLbT.Free;
   if Assigned(fBMP)
   then begin
      fBMP.Free;
      fBMP := nil;
   end;
   if Assigned(fImg)    then fImg.Free;
   if Assigned(fPanel)  then fPanel.Free;
   inherited Free;
end;
// --------------------------------------------------------------------
//
// --------------------------------------------------------------------
//       
procedure TOscillograph.SetFileNumPoint();
var wStr  : string;
    wNum  : integer;
begin
   if InputQuery('      ',
                 '    : ' + IntToStr(fFileNumPnt),
                 wStr)
   then begin
      if TryStrToInt(wStr, wNum)
      then begin
           fFileNumPnt := wNum;
      end
      else MessageDlg ('   '
           +  #13#10 + '    ...'
           +  #13#10 + '   .  ',
           mtInformation, [mbOk], 0);
   end;
   ShowTitle();  //  
end;
// --------------------------------------------------------------------
//    
procedure TOscillograph.MenuClick(Sender : TObject);
var Item       : TMenuItem;

begin
   Item  :=  TMenuItem(Sender);
   case Item.Tag of
   1 : begin  //  
          if not Item.Checked
          then begin
             fAlarmOn := True;
             Item.Checked := True;
          end
          else begin
             fAlarmOn := False;
             Item.Checked := False;
          end;
       end;
   2 : begin //  
          if not Item.Checked
          then begin
             fDigit := True;
             Item.Checked := True;
          end
          else begin
             fDigit := False;
             Item.Checked := False;
          end;
       end;
   3 : begin  // 
          if not Item.Checked
          then begin
             fdYShow := True;
             Item.Checked := True;
          end
          else begin
             fdYShow := False;
             Item.Checked := False;
          end;
       end;
   4 : begin  //       
          SetFileNumPoint();
       end;
   5 : begin  //     
          fRecording := True;
       end;
   else begin end;
   end;
   //  
   ShowTitle();
end;
// --------------------------------------------------------------------
procedure TOscillograph.SetTitle(RqTitle : string);
begin
   if Assigned(fLbT)
   then begin
       fTitle := '  ' + RqTitle;
       fLbT.Caption := fTitle;
       //  
       ShowTitle();
   end;
end;
// --------------------------------------------------------------------
//      
procedure TOscillograph.SetYMax(RqYMax : extended);
begin
   if RqYMax < 1e-10
   then fYMax := 1e-10
   else fYMax := RqYMax;
   fGridY     := fYMax / 5;
   ImageClear(fFonColor);
end;
// --------------------------------------------------------------------
//      
procedure TOscillograph.SetXMax(RqXMax : integer);
begin
  if RqXMax < 8
  then fBeam1.QueueMaxCount := 8
  else fBeam1.QueueMaxCount := RqXMax;
  ImageClear(fFonColor);
end;
// --------------------------------------------------------------------
//         Y
function YFormatString(RqVal : extended): string;
begin
   Result := '%5.5f';
   if (RqVal > 10000)
   then begin Result := '%5.0f'; Exit; end;
   if (RqVal >= 1000)
   then begin Result := '%5.1f'; Exit; end;
   if (RqVal >= 100)
   then begin Result := '%5.2f'; Exit; end;
   if (RqVal >= 10)
   then begin Result := '%5.3f'; Exit; end;
   if (RqVal >= 1)
   then begin Result := '%5.4f'; Exit; end;
   if (RqVal >= 0.1)
   then begin Result := '%5.5f'; Exit; end;
end;
// --------------------------------------------------------------------
//      
procedure TOscillograph.ShowGrids();
var wY     : extended;
    wX     : integer;
    wH     : integer;
    wFrmStr : string;
begin
    with fBMP.Canvas
    do begin
       Pen.Color  := fGridColor;
       Font.Color := fTextColor;
       Font.Name  := 'Tahoma';
       Font.Size  := 8;
       wFrmStr := YFormatString(fYMax);
       // ----------------------
       //    X
       wX := 0;
       repeat
          MoveTo(SubScrLen + wX, 0,);
          LineTo(SubScrLen + wX, fBMP.Height);
          wX := wX + fGridX;
       until (wX > fBMP.Width);
       // ----------------------
       //    Y
       //  
       wH := CalcY(0);
       MoveTo(0, wH);
       LineTo(fBMP.Width - RightOff,wH);
       TextOut(4, wH-6, '0');
       //  
       Pen.Color := fGridColor;
       //    
       wY := - fGridY;
       repeat
          wH := CalcY(wY);
          MoveTo(0, wH);
          LineTo(fBMP.Width - RightOff,wH);
          if wY > - fYMax
          then TextOut(4, wH-6, Format(wFrmStr,[wY]));
          wY := wY - fGridY;
       until (wY < - fYMax);
       //    
       wY := fGridY;
       repeat
          wH := CalcY(wY);
          MoveTo(0, wH);
          LineTo(fBMP.Width - RightOff,wH);
          if wY <  fYMax
          then TextOut(4, wH-6, Format(wFrmStr,[wY]));
          wY := wY + fGridY;
       until (wY > fYMax);
    end;
end;
// --------------------------------------------------------------------
//     fImg
procedure TOscillograph.BmpToImg ();
var wImgRect : TRect;
begin
  if (not Assigned(fImg)) or (not Assigned(fBMP)) then Exit;
  //     fImg
  wImgRect := Rect(0, 0, fImg.Width, fImg.Height);
  fImg.Canvas.CopyRect(wImgRect,fBMP.Canvas,wImgRect);
end;
// --------------------------------------------------------------------
//     
procedure TOscillograph.ImageClear(RqColor : TColor);
begin
  with fBMP.Canvas do
  begin
    Brush.Color := RqColor;
    Brush.Style := bsSolid;
    FillRect(Rect(0,0, fBMP.Width, fBMP.Height));
  end;
  //    
  fXCount := 0;
end;
// --------------------------------------------------------------------
//  X -   
function TOscillograph.CalcX() : integer;
begin
  Result := SubScrLen
         + Trunc(((fBMP.Width - SubScrLen - RightOff)
         / fBeam1.QueueMaxCount) * fXCount);
end;
// --------------------------------------------------------------------
//  Y -   
function TOscillograph.CalcY(RqData : extended) : integer;
begin
  Result := (fBMP.Height div 2);
  Result := Result - Trunc((Result / fYMax) * RqData);
end;
// --------------------------------------------------------------------
//    
procedure TOscillograph.ShowFirstPoint(RqData : extended);
begin
   //   
  ImageClear(fFonColor);
  ShowGrids();
  //   
  fBMP.Canvas.MoveTo(CalcX(),CalcY(RqData));
  fYPrev  := RqData;
  fXCount := fXCount + 1;
end;
// --------------------------------------------------------------------
//    
function TOscillograph.GetBeamColor(RqData : extended): TColor;
begin
   Result := fBeamColor;
   if fAlarmOn
   then begin
     //     
     if Abs(RqData) >= fY1Alarm then Result := fc1Alarm;
     if Abs(RqData) >= fY2Alarm then Result := fc2Alarm;
   end;
end;
// --------------------------------------------------------------------
//    
procedure TOscillograph.ShowNextPoint(RqData : extended);
begin
   if fDigit
   then begin
       fBMP.Canvas.Pen.Color := GetBeamColor(fYPrev);
       fBMP.Canvas.LineTo(CalcX(),CalcY(fYPrev));
   end;
   fBMP.Canvas.Pen.Color := GetBeamColor(RqData);
   fBMP.Canvas.LineTo(CalcX(),CalcY(RqData));
   fYPrev  := RqData;
   fXCount := fXCount + 1;
end;

// --------------------------------------------------------------------
//  
procedure TOscillograph.ShowTitle();
const RXB  = 8;
      REL  = 14;
      BLNK = 4;
begin
  if fRecording
  then begin
     //   fLbT.Caption  Recording 
     with fLbT.Canvas
     do begin
        //  fLbT
        Brush.Color := fLbT.Color;
        FillRect(Rect(1,1,fLbT.Width - 1,fLbT.Height - 1));
        // 
        Brush.Style := bsSolid;
        Brush.Color := clAqua;
        Ellipse(RXB, 1, RXB + REL, 1 + REL);
        Brush.Style := bsClear;
        TextOut(RXB + REL + BLNK, 1, 'REC // ' + fTitle);
     end;
  end
  else begin
     //   fLbT.Caption
     fLbT.Caption := fTitle;
     fLbT.Repaint;
  end;
end;

// --------------------------------------------------------------------
//       
procedure TOscillograph.DialogSaveToFile();
var wName   : string;
    wExt    : string;
    wDialog : TSaveDialog;
    wFileExt : string;     //   
begin
  wExt   := '.txt';
  wDialog := TSaveDialog.Create(nil);
  //  
  wDialog.Filter := 'Oscillogramma (*'
                    + LowerCase(wExt) + ')|*' + UpperCase(wExt);
  //   
  if wDialog.Execute
  then begin
     wName := wDialog.FileName;
     wFileExt  := UpperCase(ExtractFileExt(wName));
     //     ,    
     if not (wFileExt = UpperCase(wExt))
     then wName := wName + LowerCase(wExt);
     //  
     try
        fListFilePnt.SaveToFile(wName);
     except
        MessageDlg ('    : '
                    +  #13#10 + wName,
                    mtError, [mbOk], 0);
     end;
  end;
  wDialog.Free;
end;
// --------------------------------------------------------------------
//      
procedure TOscillograph.AddDataToList(RqData : extended);
begin
   if not fRecording then Exit;
   if fFileCount = 0                 //    
   then begin
      //  
      with fListFilePnt
      do begin
         Clear;
         Add('//  : ' + fTitle);
         Add('//     : ' + DateTimeToStr(Now));
         Add('// ' + #09 + '');
         Add('   ' + IntToStr(fFileCount)
             + #09 + FloatToStr(RqData));
         Inc(fFileCount);
      end;
   end
   else begin
      //   
      fListFilePnt.Add('   ' + IntToStr(fFileCount)
                       + #09 + FloatToStr(RqData));
      Inc(fFileCount);
      //     
      if fFileCount >= fFileNumPnt
      then begin
         fRecording := False;       //  
         //       
         DialogSaveToFile();
         //       
         fListFilePnt.Clear;
         fFileCount := 0;          //     
         ShowTitle();              //  
      end;
   end;
end;
// --------------------------------------------------------------------
//  PUBLIC 
// --------------------------------------------------------------------
//     
procedure TOscillograph.AddNewData(RqData : extended);
begin
   if Assigned (fBeam1) then fBeam1.AddData(RqData);
   if Assigned (fLbV)
   then begin
     if not fdYShow
     then fLbV.Caption := '   : '
                       + Format('%6.3f',[RqData]) + ' ' + YMes
     else fLbV.Caption := '   : '
                       + Format('%6.3f',[RqData-fYPrev]) + ' ' + YMes;
   end;
   //      
   AddDataToList(RqData);
end;
// --------------------------------------------------------------------
//    
procedure TOscillograph.ShowBeam();
begin
   if Assigned (fBeam1)
   then begin
      if fBeam1.SetCurrentItem('H')
      then begin
         ShowFirstPoint(fBeam1.GetCurrentData());
         while fBeam1.SetCurrentItem('N')
         do ShowNextPoint(fBeam1.GetCurrentData());
      end;
      //     fImg
      BmpToImg ();
   end;
end;

// --------------------------------------------------------------------
// 03.01.2018
//  
procedure TOscillograph.Clear();
begin
   fBeam1.ClearQueue();   //   
   fRecording := False;   //         
   fListFilePnt.Clear;    //       
   fFileCount := 0;       //     
   fLbV.Caption :='';     //   
   ShowTitle();           //  
   // ---------------------
   ImageClear(fFonColor);
   ShowGrids();
   //     fImg
   BmpToImg ();
end;

// ====================================================================
// ====================================================================

end.
